/**
 * server.c
 *
 * Програма моделює роботу послідовного сервера, який здійснює обмін
 * даними з клієнтами через датаграмний сокет в комунікаційному домені
 * AF_INET. Отримавши черговий клієнтський запит, сервер просто повертає
 * його назад клієнту (працює як сервер служби "Луна"). Сервер
 * прив'язується до всіх IP-адрес свого вузла і порта, заданого константою
 * SERVER_PORTNUM
 *
 */

#include "myecho.h"

#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>

/*
 * Журнал помилок
 */
#define ERRLOGFILE "server.err"

int main()
{
	// Дескриптор серверного сокета 
	int s;
	// Адреса сервера
	struct sockaddr_in serv_addr;
	// Значення опції SO_REUSEADDR,
	// яка встановлюється для серверного сокета
	int reuseaddr = 1;
    
	// Приєднує стандартний потік виведення повідомлень про помилки
	// до файлу ERRLOGFILE 
	if ((freopen(ERRLOGFILE, "w", stderr)) == NULL)
		exit(EXIT_FAILURE);

	// Створює датаграмний сокет в домені AF_INET
	s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s == -1) {
		perror("Error creating server socket");
		exit(EXIT_FAILURE);
	}

	// Встановлює для сокета s опцію SO_REUSEADDR
	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR,
	               (void *) &reuseaddr, (socklen_t) sizeof(int)) == -1) {
		perror("Error setting server socket address reusable");
		exit(EXIT_FAILURE);
	}

	// Розміщує в структурі serv_addr свою адресу
	memset(&serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_addr.s_addr = INADDR_ANY;	// Всі IP-адреси даного вузла
	// htons() переставляє байти свого аргумента згідно з мережевим порядком
	serv_addr.sin_port = htons(SERVER_PORTNUM);
	// Прив'язує сокет s до локальної адреси
	if (bind(s, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) == -1) {
		perror("Error binding server socket");
		exit(EXIT_FAILURE);
	}

	// Основний робочий цикл
	while (1) {
		// Адреса клієнта
		struct sockaddr_in clnt_addr;
		// Довжина адреси клієнта
		socklen_t clnt_addr_len;
		// Буфер для повідомлень
		char buf[BUFSIZE];
		ssize_t n_recved;

		// Очищає структуру для клієнтської адреси
		memset(&clnt_addr, 0, sizeof(clnt_addr));
		clnt_addr_len = sizeof(clnt_addr);

		// Отримує датаграму
		n_recved = recvfrom(s, buf, sizeof(buf), 0,
		                    (struct sockaddr *) &clnt_addr,
		                    &clnt_addr_len);
		if (n_recved > 0) {
			// Відправляє датаграму назад
			if (sendto(s, buf, n_recved, 0, 
			           (struct sockaddr *) &clnt_addr,
			           clnt_addr_len) == -1) 
				perror("Error sending datagram to client");
		} else if (n_recved < 0)
			perror("Error receiving datagram from client");
	}

	exit(EXIT_SUCCESS);
}
